/*
 * Decompiled with CFR 0.152.
 */
package com.ardor3d.image.util.dds;

import com.ardor3d.image.Image;
import com.ardor3d.image.ImageDataFormat;
import com.ardor3d.image.PixelDataType;
import com.ardor3d.image.util.ImageLoader;
import com.ardor3d.image.util.ImageUtils;
import com.ardor3d.image.util.dds.DdsHeader;
import com.ardor3d.image.util.dds.DdsHeaderDX10;
import com.ardor3d.image.util.dds.DdsUtils;
import com.ardor3d.util.LittleEndianDataInput;
import com.ardor3d.util.geom.BufferUtils;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.logging.Logger;

public class DdsLoader
implements ImageLoader {
    private static final Logger logger = Logger.getLogger(DdsLoader.class.getName());

    @Override
    public Image load(InputStream is, boolean flipVertically) throws IOException {
        try (LittleEndianDataInput in = new LittleEndianDataInput(is);){
            int dwMagic = in.readInt();
            if (dwMagic != DdsUtils.getInt("DDS ")) {
                throw new Error("Not a dds file.");
            }
            logger.finest("Reading DDS file.");
            DdsImageInfo info = new DdsImageInfo();
            info.flipVertically = flipVertically;
            info.header = DdsHeader.read(in);
            info.headerDX10 = info.header.ddpf.dwFourCC == DdsUtils.getInt("DX10") ? DdsHeaderDX10.read(in) : null;
            Image image = new Image();
            image.setWidth(info.header.dwWidth);
            image.setHeight(info.header.dwHeight);
            DdsLoader.updateDepth(image, info);
            DdsLoader.populateImage(image, info, in);
            Image image2 = image;
            return image2;
        }
    }

    private static final void updateDepth(Image image, DdsImageInfo info) {
        if (DdsUtils.isSet(info.header.dwCaps2, 512)) {
            int depth = 0;
            if (DdsUtils.isSet(info.header.dwCaps2, 1024)) {
                ++depth;
            }
            if (DdsUtils.isSet(info.header.dwCaps2, 2048)) {
                ++depth;
            }
            if (DdsUtils.isSet(info.header.dwCaps2, 4096)) {
                ++depth;
            }
            if (DdsUtils.isSet(info.header.dwCaps2, 8192)) {
                ++depth;
            }
            if (DdsUtils.isSet(info.header.dwCaps2, 16384)) {
                ++depth;
            }
            if (DdsUtils.isSet(info.header.dwCaps2, 32768)) {
                ++depth;
            }
            if (depth != 6) {
                throw new Error("Cubemaps without all faces defined are not currently supported.");
            }
            image.setDepth(depth);
        } else {
            image.setDepth(info.header.dwDepth > 0 ? info.header.dwDepth : 1);
        }
    }

    private static final void populateImage(Image image, DdsImageInfo info, LittleEndianDataInput in) throws IOException {
        boolean alpha;
        boolean lum;
        boolean alphaPixels;
        boolean rgb;
        boolean compressedFormat;
        block22: {
            block20: {
                int fourCC;
                block25: {
                    block24: {
                        block23: {
                            block21: {
                                int flags = info.header.ddpf.dwFlags;
                                compressedFormat = DdsUtils.isSet(flags, 4);
                                rgb = DdsUtils.isSet(flags, 64);
                                alphaPixels = DdsUtils.isSet(flags, 1);
                                lum = DdsUtils.isSet(flags, 131072);
                                alpha = DdsUtils.isSet(flags, 2);
                                if (!compressedFormat) break block20;
                                fourCC = info.header.ddpf.dwFourCC;
                                if (fourCC != DdsUtils.getInt("DXT1")) break block21;
                                info.bpp = 4;
                                logger.finest("DDS format: DXT1A");
                                image.setDataFormat(ImageDataFormat.PrecompressedDXT1A);
                                break block22;
                            }
                            if (fourCC != DdsUtils.getInt("DXT3")) break block23;
                            logger.finest("DDS format: DXT3");
                            info.bpp = 8;
                            image.setDataFormat(ImageDataFormat.PrecompressedDXT3);
                            break block22;
                        }
                        if (fourCC != DdsUtils.getInt("DXT5")) break block24;
                        logger.finest("DDS format: DXT5");
                        info.bpp = 8;
                        image.setDataFormat(ImageDataFormat.PrecompressedDXT5);
                        break block22;
                    }
                    if (fourCC != DdsUtils.getInt("DX10")) break block25;
                    switch (info.headerDX10.dxgiFormat) {
                        case DXGI_FORMAT_BC4_UNORM: {
                            logger.finest("DXGI format: BC4_UNORM");
                            info.bpp = 4;
                            image.setDataFormat(ImageDataFormat.PrecompressedLATC_L);
                            break block22;
                        }
                        case DXGI_FORMAT_BC5_UNORM: {
                            logger.finest("DXGI format: BC5_UNORM");
                            info.bpp = 8;
                            image.setDataFormat(ImageDataFormat.PrecompressedLATC_LA);
                            break block22;
                        }
                        default: {
                            throw new Error("dxgiFormat not supported: " + (Object)((Object)info.headerDX10.dxgiFormat));
                        }
                    }
                }
                if (fourCC == DdsUtils.getInt("DXT2")) {
                    logger.finest("DDS format: DXT2");
                    throw new Error("DXT2 is not supported.");
                }
                if (fourCC == DdsUtils.getInt("DXT4")) {
                    logger.finest("DDS format: DXT4");
                    throw new Error("DXT4 is not supported.");
                }
                throw new Error("unsupported compressed dds format found (" + fourCC + ")");
            }
            image.setDataType(PixelDataType.UnsignedByte);
            info.bpp = info.header.ddpf.dwRGBBitCount;
            if (rgb) {
                if (alphaPixels) {
                    logger.finest("DDS format: uncompressed rgba");
                    image.setDataFormat(ImageDataFormat.RGBA);
                } else {
                    logger.finest("DDS format: uncompressed rgb ");
                    image.setDataFormat(ImageDataFormat.RGB);
                }
            } else if (lum || alphaPixels) {
                if (lum && alphaPixels) {
                    logger.finest("DDS format: uncompressed LumAlpha");
                    image.setDataFormat(ImageDataFormat.LuminanceAlpha);
                } else if (lum) {
                    logger.finest("DDS format: uncompressed Lum");
                    image.setDataFormat(ImageDataFormat.Luminance);
                } else if (alpha) {
                    logger.finest("DDS format: uncompressed Alpha");
                    image.setDataFormat(ImageDataFormat.Alpha);
                }
            } else {
                throw new Error("unsupported uncompressed dds format found.");
            }
        }
        info.calcMipmapSizes(compressedFormat);
        image.setMipMapByteSizes(info.mipmapByteSizes);
        int totalSize = 0;
        for (int size : info.mipmapByteSizes) {
            totalSize += size;
        }
        ArrayList<ByteBuffer> imageData = new ArrayList<ByteBuffer>();
        for (int i = 0; i < image.getDepth(); ++i) {
            if (compressedFormat) {
                imageData.add(DdsLoader.readDXT(in, totalSize, info, image));
                continue;
            }
            if (!rgb && !lum && !alpha) continue;
            imageData.add(DdsLoader.readUncompressed(in, totalSize, rgb, lum, alpha, alphaPixels, info, image));
        }
        image.setData(imageData);
    }

    static final ByteBuffer readDXT(LittleEndianDataInput in, int totalSize, DdsImageInfo info, Image image) throws IOException {
        int mipWidth = info.header.dwWidth;
        int mipHeight = info.header.dwHeight;
        ByteBuffer buffer = BufferUtils.createByteBuffer(totalSize);
        for (int mip = 0; mip < info.header.dwMipMapCount; ++mip) {
            byte[] data = new byte[info.mipmapByteSizes[mip]];
            in.readFully(data);
            if (!info.flipVertically) {
                buffer.put(data);
                continue;
            }
            byte[] flipped = DdsUtils.flipDXT(data, mipWidth, mipHeight, image.getDataFormat());
            buffer.put(flipped);
            mipWidth = Math.max(mipWidth / 2, 1);
            mipHeight = Math.max(mipHeight / 2, 1);
        }
        buffer.rewind();
        return buffer;
    }

    private static ByteBuffer readUncompressed(LittleEndianDataInput in, int totalSize, boolean useRgb, boolean useLum, boolean useAlpha, boolean useAlphaPixels, DdsImageInfo info, Image image) throws IOException {
        int redLumShift = DdsUtils.shiftCount(info.header.ddpf.dwRBitMask);
        int greenShift = DdsUtils.shiftCount(info.header.ddpf.dwGBitMask);
        int blueShift = DdsUtils.shiftCount(info.header.ddpf.dwBBitMask);
        int alphaShift = DdsUtils.shiftCount(info.header.ddpf.dwABitMask);
        int sourcebytesPP = info.header.ddpf.dwRGBBitCount / 8;
        int targetBytesPP = ImageUtils.getPixelByteSize(image.getDataFormat(), image.getDataType());
        ByteBuffer dataBuffer = BufferUtils.createByteBuffer(totalSize);
        int mipWidth = info.header.dwWidth;
        int mipHeight = info.header.dwHeight;
        int offset = 0;
        for (int mip = 0; mip < info.header.dwMipMapCount; ++mip) {
            for (int y = 0; y < mipHeight; ++y) {
                for (int x = 0; x < mipWidth; ++x) {
                    byte[] b = new byte[sourcebytesPP];
                    in.readFully(b);
                    int i = DdsUtils.getInt(b);
                    byte redLum = (byte)((i & info.header.ddpf.dwRBitMask) >> redLumShift);
                    byte green = (byte)((i & info.header.ddpf.dwGBitMask) >> greenShift);
                    byte blue = (byte)((i & info.header.ddpf.dwBBitMask) >> blueShift);
                    byte alpha = (byte)((i & info.header.ddpf.dwABitMask) >> alphaShift);
                    if (info.flipVertically) {
                        dataBuffer.position(offset + ((mipHeight - y - 1) * mipWidth + x) * targetBytesPP);
                    }
                    if (useAlpha) {
                        dataBuffer.put(alpha);
                        continue;
                    }
                    if (useLum) {
                        if (useAlphaPixels) {
                            dataBuffer.put(redLum).put(alpha);
                            continue;
                        }
                        dataBuffer.put(redLum);
                        continue;
                    }
                    if (!useRgb) continue;
                    if (useAlphaPixels) {
                        dataBuffer.put(redLum).put(green).put(blue).put(alpha);
                        continue;
                    }
                    dataBuffer.put(redLum).put(green).put(blue);
                }
            }
            offset += mipWidth * mipHeight * targetBytesPP;
            mipWidth = Math.max(mipWidth / 2, 1);
            mipHeight = Math.max(mipHeight / 2, 1);
        }
        return dataBuffer;
    }

    private static final class DdsImageInfo {
        boolean flipVertically;
        int bpp = 0;
        DdsHeader header;
        DdsHeaderDX10 headerDX10;
        int[] mipmapByteSizes;

        private DdsImageInfo() {
        }

        void calcMipmapSizes(boolean compressed) {
            int width = this.header.dwWidth;
            int height = this.header.dwHeight;
            int size = 0;
            this.mipmapByteSizes = new int[this.header.dwMipMapCount];
            for (int i = 0; i < this.header.dwMipMapCount; ++i) {
                size = compressed ? (width + 3) / 4 * ((height + 3) / 4) * this.bpp * 2 : width * height * this.bpp / 8;
                this.mipmapByteSizes[i] = (size + 3) / 4 * 4;
                width = Math.max(width / 2, 1);
                height = Math.max(height / 2, 1);
            }
        }
    }
}

